﻿using System;
using System.Collections;
using System.Collections.Generic;
using System.Collections.ObjectModel;
using UnityEngine;

//MissionsManager is a singleton class responsible for generating and starting missions, as well as providing an API interface
public class MissionsManager : MonoBehaviour
{
    public List<Vector3Int> MissionBuildingPositions { get; private set; } = new List<Vector3Int>();    //Positions of buildings we've "attached" a spawned mission to
    public bool IsOnMission { get; private set; } = false;
    public Dictionary<string, Mission> GeneratedMissions { get; private set; } = new Dictionary<string, Mission>();

    private int _PreviousChunkX;
    private int _PreviousChunkY;
    private float _TimeSpentInChunk;
    private bool _SpawnedChunkMissions;
    private Mission _CurrentMission = null;

    //Singleton
    private static MissionsManager _Instance;

    public static MissionsManager Instance
    {
        get
        {
            if (_Instance == null)
            {
                _Instance = FindObjectOfType<MissionsManager>();
            }

            return _Instance;
        }
    }

    /// <summary>
    /// Shows the info for a specific mission in the UI
    /// </summary>
    /// <param name="mission">The mission to show info for</param>
    public void ShowMission(Mission mission)
    {
        GameManager.Instance.MissionInfoMenu.Show(mission);
    }

    /// <summary>
    /// Starts a mission with the specified difficulty
    /// </summary>
    /// <param name="mission">The mission to start</param>
    /// <param name="difficulty">The difficulty of the mission</param>
    public void StartMission(Mission mission, Constants.MissionDifficulties difficulty)
    {
        //Let's despawn our generated missions and set our current one
        DespawnMissions();
        _CurrentMission = mission;

        //Let's pick and play a random mission BGM file
        GameManager.Instance.CurrentMusicFile = AudioManager.Instance.AudioGroups["MissionsBGMGroup"].GetRandomFile();
        AudioManager.Instance.PlayFile(GameManager.Instance.CurrentMusicFile);

        //Update our state variables
        IsOnMission = true;
        _SpawnedChunkMissions = false;
        _TimeSpentInChunk = 0.0f;

        //Now let's start
        _CurrentMission.StartMission(difficulty);
    }

    /// <summary>
    /// Ends the current mission
    /// </summary>
    /// <param name="passed">Did we pass the mission?</param>
    /// <param name="showResult">Should we show the result UI?</param>
    public void EndMission(bool passed, bool showResult = true)
    {
        //End the mission, passing whether we passed or failed
        _CurrentMission.EndMission(passed);

        if (showResult)
        {
            //If we are going to show the result, show the correct overlay for whether or not we passed
            if(passed)
            {
                GameManager.Instance.MissionResultsOverlayController.ShowPassed(_CurrentMission.GetRewardText());
            }

            else
            {
                GameManager.Instance.MissionResultsOverlayController.ShowFailed();
            }
        }

        else
        {
            //We're not showing the result so restart the music right away
            GameManager.Instance.RestartAmbientMusic();
        }

        //Reset our state variables
        _CurrentMission = null;
        IsOnMission = false;
        _SpawnedChunkMissions = false;
        _TimeSpentInChunk = 0.0f;
    }

    /// <summary>
    /// Spawns a random selection of missions in the desired chunk
    /// </summary>
    /// <param name="chunkX">The X coordinate of the chunk</param>
    /// <param name="chunkY">The Y coordinate of the chunk</param>
    private void SpawnMissions(int chunkX, int chunkY)
    {
        GeneratedMissions.Clear();
        MissionBuildingPositions.Clear();

        List<int> instanceIDs = new List<int>();

        foreach (var building in BuildingsManager.Instance.RoadBuildings)
        {
            GameManager.Instance.RoadsTilemap.GetTile(building.Key);

            if (building.Value.ChunkX == chunkX && building.Value.ChunkY == chunkY)
            {
                if (!instanceIDs.Contains(building.Value.BuildingGameObject.GetInstanceID()))
                {
                    //Found a building we don't already know about in the chunk, add it to the list
                    instanceIDs.Add(building.Value.BuildingGameObject.GetInstanceID());
                    MissionBuildingPositions.Add(building.Key);
                }
            }
        }

        MissionBuildingPositions.Shuffle(); //Shuffle the positions around

        for(int i = 0; i < ConfigurationManager.Instance.Missions.MissionsPerChunk; i++)
        {
            //Now loop through and generate however many missions we need
            if (MissionBuildingPositions.Count >= i + 1)
            {
                string thisMissionIconID = "Mission " + ((i + 1).ToString());
                Vector3Int thisMissionBuildingLocation = MissionBuildingPositions[i];

                Mission generatedMission = GenerateMission(thisMissionBuildingLocation);
                generatedMission.SpawnMinimapIcon(thisMissionIconID);
                GeneratedMissions[thisMissionIconID] = generatedMission;
            }
        }
    }

    /// <summary>
    /// Despawns all generated missions
    /// </summary>
    private void DespawnMissions()
    {
        foreach(string missionIconID in GeneratedMissions.Keys)
        {
            MinimapManager.Instance.RemoveIcon(missionIconID);
        }

        GeneratedMissions.Clear();
        MissionBuildingPositions.Clear();
    }

    /// <summary>
    /// Generates a random mission at the desired building spawn point
    /// </summary>
    /// <param name="missionSpawnPos">The spawn position of the building the mission is "attached" to</param>
    /// <returns></returns>
    private Mission GenerateMission(Vector3Int missionSpawnPos)
    {
        Mission generatedMission = null;

        Randomizer.Regenerate();

        //Generate a mission based on RNG
        switch(Randomizer.RNG.Next(0, 3))
        {
            case 0:
                generatedMission = new TimeTrialMission(missionSpawnPos);
                break;

            case 1:
                generatedMission = new CollectionMission(missionSpawnPos);
                break;

            case 2:
                generatedMission = new WagerRaceMission(missionSpawnPos);
                break;
        }

        return generatedMission;
    }

    private void Update()
    {
        if (IsOnMission && _CurrentMission != null)
        {
            _CurrentMission.Update();   //We're on a mission so update it
        }

        else if(!IsOnMission)
        {
            //We're not on a mission so let's get the chunk the player is in
            int roundedPlayerX = Convert.ToInt32(Math.Floor(GameManager.Instance.PlayerCarGameObject.transform.position.x / ConfigurationManager.Instance.Missions.MissionSpawnChunkSize) * ConfigurationManager.Instance.Missions.MissionSpawnChunkSize);
            int roundedPlayerY = Convert.ToInt32(Math.Floor(GameManager.Instance.PlayerCarGameObject.transform.position.y / ConfigurationManager.Instance.Missions.MissionSpawnChunkSize) * ConfigurationManager.Instance.Missions.MissionSpawnChunkSize);

            int currentChunkX = Convert.ToInt32(roundedPlayerX / ConfigurationManager.Instance.Missions.MissionSpawnChunkSize);
            int currentChunkY = Convert.ToInt32(roundedPlayerY / ConfigurationManager.Instance.Missions.MissionSpawnChunkSize);

            if (currentChunkX == _PreviousChunkX && currentChunkY == _PreviousChunkY)
            {
                //We're still in the same chunk
                _TimeSpentInChunk += (Time.deltaTime * 1000.0f);

                if (_TimeSpentInChunk >= ConfigurationManager.Instance.Missions.MissionSpawnChunkDelayTime && !_SpawnedChunkMissions)
                {
                    //Enough time has been spent in the chunk, time to spawn
                    SpawnMissions(currentChunkX, currentChunkY);
                    _SpawnedChunkMissions = true;
                }
            }

            else
            {
                //We left the chunk so despawn and reset
                DespawnMissions();
                _TimeSpentInChunk = 0.0f;
                _SpawnedChunkMissions = false;
            }

            _PreviousChunkX = currentChunkX;
            _PreviousChunkY = currentChunkY;
        }
    }
}
